# SSE 心跳保活机制改进方案

## 问题描述
之前对话返回信息过长时，会因为流式响应超时（60秒无消息）而显示"[错误] 流式输出超时，请重试"，导致SSE连接被关闭。

## 解决方案

### 前端改进 (ChatArea.vue)

#### 1. 改进超时检测机制
- **原来**: 简单的60秒全局超时，无任何数据到达就关闭
- **现在**: 使用心跳保活机制，定期检查是否收到心跳

```typescript
// 关键参数
const HEARTBEAT_TIMEOUT = 60000;      // 60秒无心跳则为超时
const HEARTBEAT_CHECK_INTERVAL = 5000; // 每5秒检查一次
let lastHeartbeatTime = Date.now();    // 记录最后一次心跳时间
```

#### 2. 新增心跳事件处理
在 `processSSELine` 函数中新增 heartbeat case：
```typescript
case "heartbeat":
  // 收到心跳事件，重置超时计时器
  resetStreamTimeout();
  // 心跳事件本身不处理，只用于保活连接
  console.debug("[心跳] 收到心跳事件，连接保活");
  return false;
```

#### 3. 改进的超时判断逻辑
```typescript
const resetStreamTimeout = () => {
  clearStreamTimeout();
  lastHeartbeatTime = Date.now(); // 更新最后心跳时间
  streamTimeoutTimer = setTimeout(() => {
    if (!isStreamComplete) {
      // 检查是否在指定时间内收到过心跳或数据
      const timeSinceLastHeartbeat = Date.now() - lastHeartbeatTime;
      if (timeSinceLastHeartbeat >= HEARTBEAT_TIMEOUT) {
        // 真正的超时，关闭连接
        isStreamComplete = true;
        reader.cancel();
        // ... 显示超时错误
      } else {
        // 还没超时，继续检查
        resetStreamTimeout();
      }
    }
  }, HEARTBEAT_CHECK_INTERVAL);
};
```

**工作原理**：
1. 每当收到token、心跳或其他数据时，重置超时计时器并更新`lastHeartbeatTime`
2. 每5秒检查一次是否超时
3. 只有当最后一次心跳/数据距现在超过60秒时，才真正认为超时并关闭连接
4. 否则继续检查，保持连接活跃

---

### 后端改进 (UserSseService.java)

#### 1. 调整心跳发送频率
- **原来**: 每30秒发送一次心跳
- **现在**: 每20秒发送一次心跳

```java
}, 20, 20, TimeUnit.SECONDS); // 每20秒发送一次心跳，确保前端60秒超时前至少收到2次心跳
```

**原因**: 确保在前端60秒超时前，至少能收到2次心跳，增加可靠性

#### 2. 增强心跳日志
```java
long heartbeatTimestamp = System.currentTimeMillis();
emitter.send(SseEmitter.event().name("heartbeat").data(heartbeatTimestamp));
log.debug("[心跳] 成功发送心跳事件，时间戳: {}", heartbeatTimestamp);
```

#### 3. 心跳机制的完整生命周期
- **启动**: 创建连接时调用 `startHeartbeat()`
- **运行**: 每20秒检查一次连接有效性，如果有效则发送心跳
- **停止**: 在连接完成/超时/错误时自动取消心跳任务

```java
// 注册回调，在连接完成时取消心跳任务
emitter.onCompletion(() -> {
    if (heartbeatTask != null && !heartbeatTask.isCancelled()) {
        heartbeatTask.cancel(true);
        log.debug("SSE连接完成，心跳任务已取消");
    }
});

// 类似的处理: onTimeout(), onError()
```

---

## 工作流程

### 正常情况（消息持续到达）
```
时间轴: 0s ─── 10s ─── 20s ─── 30s ─── 40s ─── 50s ─── 60s
                │          │         │
              token      token     token
                │          │         │
          重置超时    重置超时    重置超时
           (60s)      (60s)      (60s)
```
连接保持活跃，不会超时。

### 有心跳但消息间隔长（解决长时间处理问题）
```
时间轴: 0s ─── 10s ─── 20s ─── 30s ─── 40s ─── 50s ─── 60s ─── 70s ─── 80s
       token          心跳              心跳              心跳         token
        │             │                 │                 │            │
    重置超时      重置超时         重置超时          重置超时      重置超时
     (60s)       (60s)            (60s)             (60s)        (60s)
```
心跳每20秒发送一次，保持连接活跃，即使消息处理需要很长时间。

### 真正超时的情况（心跳也断开）
```
时间轴: 0s ─── 20s ─── 40s ─── 60s ─── 70s（超时）
       token   心跳   心跳  [无更多心跳]
        │      │      │
    重置超时 重置超时 重置超时
     (60s)   (60s)   (60s)
                     ↓
              超过60秒无响应，关闭连接
```
当网络真的中断或服务器崩溃时，经过60秒无任何响应，客户端才会超时并提示用户。

---

## 关键时间参数

| 参数 | 值 | 说明 |
|------|-----|------|
| 心跳间隔（后端）| 20秒 | 后端定期向客户端发送心跳 |
| 前端超时时间 | 60秒 | 前端在60秒内无心跳/数据则超时 |
| 检查间隔（前端）| 5秒 | 前端每5秒检查一次是否超时 |
| SSE连接超时（后端）| 120秒 | Spring框架层面的连接超时 |

**设计原理**: 心跳间隔 (20s) < 前端超时时间 (60s) / 2，保证前端超时前至少收到2次心跳。

---

## 对话结束和错误处理

### 对话正常结束
1. 后端发送 `complete` 事件
2. 前端收到 `complete` 事件，调用 `clearStreamTimeout()`
3. 流式处理完成，关闭所有计时器和监听

### 发生错误时
1. 后端发送 `error` 事件
2. 前端收到 `error` 事件，调用 `clearStreamTimeout()`
3. 关闭连接和心跳检查，显示错误信息

### 心跳中断且超时
1. 前端在60秒内未收到任何心跳/数据
2. 前端认定连接超时，取消读取并显示错误
3. 用户可以点击重试按钮重新发送消息

---

## 调试

### 后端日志
```
[心跳] 成功发送心跳事件，时间戳: 1640000000000
```

### 前端日志
```
[心跳] 收到心跳事件，连接保活
[SSE完成事件] {type: "complete", ...}
```

### 超时测试
1. 故意让后端处理延迟超过60秒的请求
2. 观察是否能收到心跳事件
3. 连接应该保持活跃，不会因为消息间隔长而断开
4. 直到对话完成或心跳真的中断，才会关闭连接

---

## 总结

这个改进方案通过引入心跳保活机制，解决了以下问题：

✅ 长时间处理的对话不会因为超时而意外断开
✅ 心跳中断才会真正关闭连接（而不是任意时间无消息就关闭）
✅ 流式响应自然结束或错误发生时，及时清理资源
✅ 系统更加稳定可靠，特别是对于复杂AI任务处理

