package pangea.hiagent.common.utils;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.concurrent.Callable;

/**
 * 异步任务用户上下文装饰器
 * 用于在异步任务执行时自动传播用户认证信息
 */
@Slf4j
@Component
public class AsyncUserContextDecorator {
    
    /**
     * 包装Runnable任务，自动传播用户上下文
     * @param runnable 原始任务
     * @return 包装后的任务
     */
    public static Runnable wrapWithContext(Runnable runnable) {
        // 捕获当前线程的用户上下文
        UserContextPropagationUtil.UserContextHolder userContext = UserContextPropagationUtil.captureUserContext();
        
        return () -> {
            try {
                // 在异步线程中传播用户上下文
                UserContextPropagationUtil.propagateUserContext(userContext);
                
                // 执行原始任务
                runnable.run();
            } finally {
                // 清理当前线程的用户上下文
                UserContextPropagationUtil.clearUserContext();
            }
        };
    }
    
    /**
     * 包装Callable任务，自动传播用户上下文
     * @param callable 原始任务
     * @param <V> 返回值类型
     * @return 包装后的任务
     */
    public static <V> Callable<V> wrapWithContext(Callable<V> callable) {
        // 捕获当前线程的用户上下文
        UserContextPropagationUtil.UserContextHolder userContext = UserContextPropagationUtil.captureUserContext();
        
        return () -> {
            try {
                // 在异步线程中传播用户上下文
                UserContextPropagationUtil.propagateUserContext(userContext);
                
                // 执行原始任务
                return callable.call();
            } finally {
                // 清理当前线程的用户上下文
                UserContextPropagationUtil.clearUserContext();
            }
        };
    }
}