package pangea.hiagent.document;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.context.annotation.Lazy;

import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.model.Agent;

import java.util.List;

/**
 * 知识库初始化服务
 * 在应用启动时为各Agent创建相应的知识库
 */
@Slf4j
@Service
@Lazy
public class KnowledgeBaseInitializationService {
    
    @Autowired
    private DocumentManagementService documentManagementService;
    
    @Autowired
    private AgentService agentService;
    
    /**
     * 手动触发初始化知识库
     * 
     * 该方法不再在应用启动时自动执行，而是通过管理界面手动触发
     */
    public void initializeKnowledgeBasesManually() {
        log.info("开始初始化各Agent的知识库");
        
        try {
            // 获取所有Agent
            List<Agent> agents = agentService.listAgents();
            
            if (agents == null || agents.isEmpty()) {
                log.warn("未找到任何Agent，跳过知识库初始化");
                return;
            }
            
            // 为每个Agent创建知识库
            for (Agent agent : agents) {
                log.info("为Agent {} 创建知识库", agent.getName());
                documentManagementService.createKnowledgeBaseForAgent(agent);
            }
            
            log.info("所有Agent的知识库初始化完成");
        } catch (Exception e) {
            log.error("初始化知识库时发生错误", e);
        }
    }
}