package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.model.Agent;
import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.common.utils.LogUtils;
import pangea.hiagent.common.utils.ValidationUtils;
import pangea.hiagent.common.utils.UserUtils;

/**
 * Agent验证服务
 * 负责处理Agent的参数验证和权限检查
 */
@Slf4j
@Service
public class AgentValidationService {
    
    @Autowired
    private AgentService agentService;
    
    @Autowired
    private ChatErrorHandler chatErrorHandler;
    
    /**
     * 验证Agent存在性和用户权限
     *
     * @param agentId Agent ID
     * @param userId 用户ID
     * @param emitter SSE发射器
     * @return Agent对象，如果验证失败则返回null
     */
    public Agent validateAgentAndPermission(String agentId, String userId, SseEmitter emitter) {
        LogUtils.enterMethod("validateAgentAndPermission", agentId, userId);
        // 参数验证
        if (ValidationUtils.isBlank(agentId, "agentId")) {
            chatErrorHandler.handleChatError(emitter, "Agent ID不能为空");
            LogUtils.exitMethod("validateAgentAndPermission", "Agent ID不能为空");
            return null;
        }
        
        if (ValidationUtils.isBlank(userId, "userId")) {
            chatErrorHandler.handleChatError(emitter, "用户ID不能为空");
            LogUtils.exitMethod("validateAgentAndPermission", "用户ID不能为空");
            return null;
        }
        
        try {
            // 获取Agent信息
            Agent agent = agentService.getAgent(agentId);
            if (agent == null) {
                chatErrorHandler.handleChatError(emitter, "Agent不存在");
                LogUtils.exitMethod("validateAgentAndPermission", "Agent不存在");
                return null;
            }

            // 检查权限（可选）
            if (!agent.getOwner().equals(userId) && !UserUtils.isAdminUser(userId)) {
                chatErrorHandler.handleChatError(emitter, "无权限访问该Agent");
                LogUtils.exitMethod("validateAgentAndPermission", "无权限访问该Agent");
                return null;
            }

            LogUtils.exitMethod("validateAgentAndPermission", agent);
            return agent;
        } catch (Exception e) {
            chatErrorHandler.handleChatError(emitter, "验证Agent和权限时发生错误", e, null);
            LogUtils.exitMethod("validateAgentAndPermission", e);
            return null;
        }
    }
}