package pangea.hiagent.websocket;

import com.alibaba.fastjson2.JSON;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;

/**
 * 统计信息服务
 * 负责统计信息的收集和报告
 */
@Slf4j
public class StatisticsService {
    private final ConcurrentMap<String, Long> messageCounters;
    private final ConcurrentMap<String, Integer> userConnections;
    
    public StatisticsService(ConcurrentMap<String, Long> messageCounters, 
                           ConcurrentMap<String, Integer> userConnections) {
        this.messageCounters = messageCounters;
        this.userConnections = userConnections;
    }
    
    /**
     * 获取统计信息摘要
     * 
     * @return 包含所有统计信息的JSON字符串
     */
    public String getStatisticsSummary() {
        try {
            Map<String, Object> stats = new HashMap<>();
            stats.put("userConnections", new HashMap<>(userConnections));
            stats.put("domChanges", getCounter("domChanges"));
            stats.put("websocketMessages", getCounter("websocketMessages"));
            stats.put("errors", getCounter("errors"));
            stats.put("pageLoads", getCounter("pageLoads"));
            stats.put("navigations", getCounter("navigations"));
            stats.put("timestamp", System.currentTimeMillis());
            
            return JSON.toJSONString(stats);
        } catch (Exception e) {
            log.error("获取统计信息失败", e);
            return "{\"error\":\"获取统计信息失败\"}";
        }
    }
    
    /**
     * 重置所有统计计数器
     */
    public void resetAllCounters() {
        try {
            String[] counters = {"domChanges", "websocketMessages", "errors", "pageLoads", "navigations"};
            for (String counter : counters) {
                resetCounter(counter);
            }
            log.debug("所有统计计数器已重置");
        } catch (Exception e) {
            log.error("重置统计计数器失败", e);
        }
    }
    
    /**
     * 获取计数器值
     */
    private long getCounter(String counterName) {
        return messageCounters.getOrDefault(counterName, 0L);
    }
    
    /**
     * 重置计数器
     */
    private void resetCounter(String counterName) {
        messageCounters.put(counterName, 0L);
    }
}