package pangea.hiagent.websocket;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.socket.BinaryMessage;
import org.springframework.web.socket.WebSocketSession;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ConcurrentMap;

/**
 * 二进制消息发送服务
 * 负责通过WebSocket发送二进制消息
 */
@Slf4j
public class BinaryMessageSender {
    // 异步发送线程池（用于非阻塞发送二进制消息）
    private final ExecutorService binaryMessageExecutor;
    
    private final WebSocketConnectionManager connectionManager;
    
    public BinaryMessageSender(WebSocketConnectionManager connectionManager) {
        this.connectionManager = connectionManager;
        this.binaryMessageExecutor = Executors.newFixedThreadPool(
                Math.max(4, Runtime.getRuntime().availableProcessors() / 2),
                new ThreadFactory() {
                    private int count = 0;
                    @Override
                    public Thread newThread(Runnable r) {
                        Thread t = new Thread(r, "DomSync-BinaryMsg-" + (++count));
                        t.setDaemon(true);
                        return t;
                    }
                }
        );
    }
    
    /**
     * 发送二进制消息（大消息优化）
     * 使用BinaryMessageProtocol进行二进制编码，支持自动分片
     * 异步发送，避免在IO线程中阻塞
     * 
     * @param message 待发送的消息内容
     */
    public void sendBinaryMessage(String message) {
        // 异步提交到线程池，避免在WebSocket IO线程中阻塞
        binaryMessageExecutor.submit(() -> {
            try {
                sendBinaryMessageInternal(message);
            } catch (Exception e) {
                log.error("发送二进制消息失败", e);
            }
        });
    }
    
    /**
     * 内部方法：真正执行二进制消息发送
     * 在独立的工作线程中运行，不会阻塞WebSocket IO线程
     */
    private void sendBinaryMessageInternal(String message) {
        byte[] messageBytes = message.getBytes(StandardCharsets.UTF_8);
        int messageId = BinaryMessageProtocol.generateMessageId();
        int maxFragmentSize = BinaryMessageProtocol.MAX_FRAGMENT_SIZE;
        
        int fragmentCount = (int) Math.ceil((double) messageBytes.length / maxFragmentSize);
        
        // 检查是否有客户端连接
        int clientCount = connectionManager.getClientCount();
        
        log.debug("========== 发送二进制消息 ==========");
        log.debug("消息ID: {}, 大小: {} 字节, 分片数: {}", messageId, messageBytes.length, fragmentCount);
        log.debug("发送给客户端数量: {}", clientCount);
        
        log.debug("当前客户端连接数: {}", clientCount);
        
        if (clientCount == 0) {
            log.warn("没有客户端连接，消息将不会被发送");
            return;
        }
        
        for (int i = 0; i < fragmentCount; i++) {
            int start = i * maxFragmentSize;
            int end = Math.min(start + maxFragmentSize, messageBytes.length);
            byte[] fragmentData = Arrays.copyOfRange(messageBytes, start, end);
            
            // 创建协议头
            byte[] header = BinaryMessageProtocol.encodeHeader(
                BinaryMessageProtocol.TYPE_DATA,
                messageId,
                fragmentCount,
                i,
                BinaryMessageProtocol.ENCODING_RAW
            );
            
            // 合并头和数据
            byte[] fullMessage = BinaryMessageProtocol.buildMessage(header, fragmentData);
            
            log.debug("分片 {}/{}: 大小={} 字节, 客户端数量={}", i, fragmentCount - 1, fragmentData.length, clientCount);
            
            // 发送到所有客户端（在工作线程中执行，不会阻塞IO线程）
            BinaryMessage binaryMessage = new BinaryMessage(fullMessage);
            
            // 通过连接管理器广播消息
            connectionManager.broadcastMessage(binaryMessage, this);
        }
        
        log.debug("✓ 二进制消息发送完成: 消息ID={}, 总分片数={}, 客户端数量={}", messageId, fragmentCount, clientCount);
    }
    
    /**
     * 发送错误信息给所有客户端
     * 使用二进制协议发送错误消息
     * 异步发送，避免在IO线程中阻塞
     */
    public void sendErrorToClients(String errorMessage) {
        // 异步提交到线程池，避免在WebSocket IO线程中阻塞
        binaryMessageExecutor.submit(() -> {
            try {
                sendErrorToClientsInternal(errorMessage);
            } catch (Exception e) {
                log.error("发送错误消息失败", e);
            }
        });
    }
    
    /**
     * 内部方法：真正执行错误消息发送
     * 在独立的工作线程中运行
     * 支持大消息分片传输
     */
    private void sendErrorToClientsInternal(String errorMessage) {
        byte[] messageBytes = errorMessage.getBytes(StandardCharsets.UTF_8);
        int messageId = BinaryMessageProtocol.generateMessageId();
        int maxFragmentSize = BinaryMessageProtocol.MAX_FRAGMENT_SIZE;
        
        int fragmentCount = (int) Math.ceil((double) messageBytes.length / maxFragmentSize);
        
        log.debug("========== 发送二进制错误消息 ==========");
        log.debug("错误消息ID: {}, 大小: {} 字节, 分片数: {}", messageId, messageBytes.length, fragmentCount);
        
        for (int i = 0; i < fragmentCount; i++) {
            int start = i * maxFragmentSize;
            int end = Math.min(start + maxFragmentSize, messageBytes.length);
            byte[] fragmentData = Arrays.copyOfRange(messageBytes, start, end);
            
            // 创建错误帧协议头
            byte[] header = BinaryMessageProtocol.encodeHeader(
                BinaryMessageProtocol.TYPE_ERROR,
                messageId,
                fragmentCount,
                i,
                BinaryMessageProtocol.ENCODING_RAW
            );
            
            // 合并头和数据
            byte[] fullMessage = BinaryMessageProtocol.buildMessage(header, fragmentData);
            
            log.debug("错误分片 {}/{}: 大小={} 字节", i, fragmentCount - 1, fragmentData.length);
            
            // 发送到所有客户端
            BinaryMessage binaryMessage = new BinaryMessage(fullMessage);
            
            // 通过连接管理器广播消息
            connectionManager.broadcastMessage(binaryMessage, this);
        }
        
        log.debug("✓ 错误消息发送完成: 消息ID={}, 总分片数={}", messageId, fragmentCount);
    }
    
    /**
     * 广播消息给所有客户端
     * 统一使用二进制协议发送
     */
    public void broadcastMessage(String message) {
        // 所有消息都通过sendBinaryMessage发送
        if (message == null || message.isEmpty()) {
            return;
        }
        
        try {
            // 转换为二进制格式发送
            sendBinaryMessage(message);
        } catch (Exception e) {
            log.error("广播消息时发生异常: {}", e.getMessage(), e);
        }
    }
    
    /**
     * 销毁资源
     */
    public void destroy() {
        try {
            // 关闭线程池
            binaryMessageExecutor.shutdown();
            if (!binaryMessageExecutor.awaitTermination(5, java.util.concurrent.TimeUnit.SECONDS)) {
                binaryMessageExecutor.shutdownNow();
            }
        } catch (Exception e) {
            log.error("关闭线程池失败", e);
        }
    }
}