package pangea.hiagent.common.utils;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import java.io.Serializable;
import java.util.concurrent.Callable;

/**
 * 异步任务用户上下文装饰器
 * 用于在异步任务执行时自动传播用户认证信息
 */
@Slf4j
@Component
public class AsyncUserContextDecorator {
    
    /**
     * 用户上下文持有者类，用于在异步线程间传递认证信息
     */
    public static class UserContextHolder implements Serializable {
        private static final long serialVersionUID = 1L;
        
        private final Authentication authentication;
        
        public UserContextHolder(Authentication authentication) {
            this.authentication = authentication;
        }
        
        public Authentication getAuthentication() {
            return authentication;
        }
    }
    
    /**
     * 捕获当前线程的用户上下文
     * @return 用户上下文持有者对象
     */
    public static UserContextHolder captureUserContext() {
        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null) {
                log.debug("捕获到当前线程的用户认证信息: {}", authentication.getPrincipal());
                return new UserContextHolder(authentication);
            } else {
                log.debug("当前线程无用户认证信息");
                return null;
            }
        } catch (Exception e) {
            log.error("捕获用户上下文时发生异常", e);
            return null;
        }
    }
    
    /**
     * 将用户上下文传播到当前线程
     * @param userContextHolder 用户上下文持有者对象
     */
    public static void propagateUserContext(UserContextHolder userContextHolder) {
        try {
            if (userContextHolder != null) {
                Authentication authentication = userContextHolder.getAuthentication();
                if (authentication != null) {
                    SecurityContext context = SecurityContextHolder.createEmptyContext();
                    context.setAuthentication(authentication);
                    SecurityContextHolder.setContext(context);
                    log.debug("已将用户认证信息传播到当前线程: {}", authentication.getPrincipal());
                } else {
                    log.debug("用户上下文持有者中的认证信息为空");
                }
            } else {
                log.debug("用户上下文持有者为空");
            }
        } catch (Exception e) {
            log.error("传播用户上下文时发生异常", e);
        }
    }
    
    /**
     * 清理当前线程的用户上下文
     */
    public static void clearUserContext() {
        try {
            SecurityContextHolder.clearContext();
            log.debug("已清理当前线程的用户上下文");
        } catch (Exception e) {
            log.error("清理用户上下文时发生异常", e);
        }
    }
    
    /**
     * 包装Runnable任务，自动传播用户上下文
     * @param runnable 原始任务
     * @return 包装后的任务
     */
    public static Runnable wrapWithContext(Runnable runnable) {
        // 捕获当前线程的用户上下文
        UserContextHolder userContext = captureUserContext();
        // 同时捕获当前线程的用户ID（用于ThreadLocal传播）
        String currentUserId = UserUtils.getCurrentUserIdStatic();
        
        return () -> {
            try {
                // 在异步线程中传播用户上下文
                propagateUserContext(userContext);
                // 将用户ID设置到ThreadLocal中，增强可靠性
                if (currentUserId != null) {
                    UserUtils.setCurrentUserIdStatic(currentUserId);
                }
                
                // 执行原始任务
                runnable.run();
            } finally {
                // 清理当前线程的用户上下文
                clearUserContext();
                // 清理ThreadLocal中的用户ID
                UserUtils.clearCurrentUserIdStatic();
            }
        };
    }
    
    /**
     * 包装Callable任务，自动传播用户上下文
     * @param callable 原始任务
     * @param <V> 返回值类型
     * @return 包装后的任务
     */
    public static <V> Callable<V> wrapWithContext(Callable<V> callable) {
        // 捕获当前线程的用户上下文
        UserContextHolder userContext = captureUserContext();
        // 同时捕获当前线程的用户ID（用于ThreadLocal传播）
        String currentUserId = UserUtils.getCurrentUserIdStatic();
        
        return () -> {
            try {
                // 在异步线程中传播用户上下文
                propagateUserContext(userContext);
                // 将用户ID设置到ThreadLocal中，增强可靠性
                if (currentUserId != null) {
                    UserUtils.setCurrentUserIdStatic(currentUserId);
                }
                
                // 执行原始任务
                return callable.call();
            } finally {
                // 清理当前线程的用户上下文
                clearUserContext();
                // 清理ThreadLocal中的用户ID
                UserUtils.clearCurrentUserIdStatic();
            }
        };
    }
}