package pangea.hiagent.agent.react;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.agent.service.UserSseService;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;

/**
 * 简化的ReAct回调类
 */
@Slf4j
@Component
public class DefaultReactCallback implements ReactCallback {
    
    @Autowired
    private IWorkPanelDataCollector workPanelCollector;

    @Autowired
    private UserSseService userSseService;

    @Override
    public void onStep(ReactStep reactStep) {
        log.info("ReAct步骤触发: 类型={}, 内容摘要={}", 
            reactStep.getStepType(), 
            reactStep.getContent() != null ? 
                reactStep.getContent().substring(0, Math.min(50, reactStep.getContent().length())) : "null");
        recordReactStepToWorkPanel(reactStep);
    }
    
    private void recordReactStepToWorkPanel(ReactStep reactStep) {
        if (workPanelCollector == null) {
            return;
        }
        
        try {
            switch (reactStep.getStepType()) {
                case THOUGHT:

                    // userSseService.sendWorkPanelEvent(reactStep.getContent(), "thought");
                    // workPanelCollector.recordThinking(reactStep.getContent(), "thought");
                    log.info("[WorkPanel] 记录思考步骤: {}", 
                        reactStep.getContent().substring(0, Math.min(100, reactStep.getContent().length())));
                    break;
                case ACTION:
                    if (reactStep.getAction() != null) {
                        // 记录工具调用动作
                        String toolName = reactStep.getAction().getToolName();
                        Object parameters = reactStep.getAction().getToolArgs();
                        
                        // 记录工具调用，初始状态为pending
                        workPanelCollector.recordToolCallAction(
                            toolName, 
                            parameters,
                            null,  // 结果为空
                            "pending",  // 状态为pending
                            null   // 错误信息为空
                        );
                        
                        // 同时记录工具调用信息到日志
                        log.info("[WorkPanel] 记录工具调用: 工具={} 参数={}", toolName, parameters);
                    } else {
                        // 如果没有具体的工具信息，记录为一般动作
                        workPanelCollector.recordThinking(reactStep.getContent(), "action");
                        log.info("[WorkPanel] 记录动作步骤: {}", 
                            reactStep.getContent().substring(0, Math.min(100, reactStep.getContent().length())));
                    }
                    break;
                case OBSERVATION:
                    if (reactStep.getObservation() != null) {
                        // 检查是否有对应的动作信息
                        if (reactStep.getAction() != null) {
                            // 使用动作信息更新工具调用结果
                            workPanelCollector.recordToolCallAction(
                                reactStep.getAction().getToolName(), 
                                reactStep.getAction().getToolArgs(),
                                reactStep.getObservation().getResult(), 
                                "success",  // 状态为success
                                null        // 无错误信息
                            );
                            
                            log.info("[WorkPanel] 更新工具调用结果: 工具={} 结果摘要={}", 
                                reactStep.getAction().getToolName(), 
                                reactStep.getObservation().getResult().substring(0, Math.min(50, reactStep.getObservation().getResult().length())));
                        } else {
                            // 如果没有动作信息，记录为观察结果
                            workPanelCollector.recordThinking(reactStep.getContent(), "observation");
                            log.info("[WorkPanel] 记录观察步骤: {}", 
                                reactStep.getContent().substring(0, Math.min(100, reactStep.getContent().length())));
                        }
                    }
                    break;
                case FINAL_ANSWER:
                    workPanelCollector.recordFinalAnswer(reactStep.getContent());
                    
                    // 记录最终答案到日志
                    log.info("[WorkPanel] 记录最终答案: {}", 
                        reactStep.getContent().substring(0, Math.min(100, reactStep.getContent().length())));
                    break;
                default:
                    log.warn("未知的ReAct步骤类型: {}", reactStep.getStepType());
                    break;
            }
        } catch (Exception e) {
            log.error("记录ReAct步骤到工作面板失败", e);
            // 即使发生异常，也尝试记录错误信息到工作面板
            try {
                if (reactStep != null && reactStep.getAction() != null) {
                    workPanelCollector.recordToolCallAction(
                        reactStep.getAction().getToolName(),
                        reactStep.getAction().getToolArgs(),
                        "记录失败: " + e.getMessage(),
                        "error",
                        System.currentTimeMillis() // 使用当前时间戳作为执行时间
                    );
                }
            } catch (Exception ex) {
                log.error("记录错误信息到工作面板也失败", ex);
            }
        }
    }
}