package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

/**
 * OpenAI模型适配器
 * 实现OpenAI模型的创建和配置
 */
@Slf4j
@Component
public class OpenAiModelAdapter extends AbstractOpenAiCompatibleAdapter {
    
    private static final String OPENAI_DEFAULT_BASE_URL = "https://api.openai.com/v1";
    
    @Override
    public ChatModel createChatModel(LlmConfig config) {
        if (!validateConfig(config)) {
            log.error("OpenAI配置验证失败，配置信息: {}", config);
            throw new IllegalArgumentException("无效的OpenAI配置");
        }
        
        // 记录API密钥长度信息（脱敏显示）
        if (config.getApiKey() != null) {
            log.info("创建OpenAI ChatModel，模型名称: {}, API密钥长度: {}", config.getModelName(), config.getApiKey().length());
        } else {
            log.info("创建OpenAI ChatModel，模型名称: {}, API密钥为空", config.getModelName());
        }
        
        return createOpenAiChatModel(config, OPENAI_DEFAULT_BASE_URL);
    }
    
    @Override
    public String getProviderName() {
        return "openai";
    }
    
    @Override
    public boolean validateConfig(LlmConfig config) {
        boolean isValid = validateOpenAiConfig(config, true); // OpenAI需要API密钥
        
        if (!isValid) {
            log.warn("OpenAI配置验证失败，配置: {}", config);
        } else {
            // 额外验证模型名称是否设置
            if (config.getModelName() == null || config.getModelName().isEmpty()) {
                log.warn("OpenAI配置缺少模型名称");
                return false;
            }
            log.debug("OpenAI配置验证通过，模型名称: {}", config.getModelName());
        }
        
        return isValid;
    }
}