package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.workpanel.event.EventService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.common.utils.LogUtils;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 流式请求服务
 * 负责处理流式请求
 */
@Slf4j
@Service
public class StreamRequestService {
    
    @Autowired
    private UserSseService unifiedSseService;
    
    @Autowired
    private EventService eventService;
    
    @Autowired
    private CompletionHandlerService completionHandlerService;
    
    /**
     * 处理流式请求
     * 
     * @param emitter SSE发射器
     * @param processor Agent处理器
     * @param request Agent请求
     * @param agent Agent对象
     * @param userId 用户ID
     */
    public void handleStreamRequest(SseEmitter emitter, AgentProcessor processor, pangea.hiagent.web.dto.AgentRequest request, Agent agent, String userId) {
        LogUtils.enterMethod("handleStreamRequest", emitter, processor, request, agent, userId);
        
        // 参数验证
        if (!validateParameters(emitter, processor, request, agent, userId)) {
            return;
        }
        
        // 创建流式处理的Token消费者
        StreamTokenConsumer tokenConsumer = new StreamTokenConsumer(emitter, processor, unifiedSseService, eventService, completionHandlerService);
        // 设置上下文信息，用于保存对话记录
        tokenConsumer.setContext(agent, request, userId);
        
        // 处理流式请求，将token缓冲和事件发送完全交给处理器实现
        processor.processStreamRequest(request, agent, userId, tokenConsumer);
        LogUtils.exitMethod("handleStreamRequest", "处理完成");
    }
    
    /**
     * 验证所有必需参数
     * 
     * @param emitter SSE发射器
     * @param processor Agent处理器
     * @param request Agent请求
     * @param agent Agent对象
     * @param userId 用户ID
     * @return 验证是否通过
     */
    private boolean validateParameters(SseEmitter emitter, AgentProcessor processor, pangea.hiagent.web.dto.AgentRequest request, Agent agent, String userId) {
        return emitter != null && processor != null && request != null && agent != null && userId != null && !userId.isEmpty();
    }
    
    /**
     * 流式处理的Token消费者实现
     * 用于处理来自Agent处理器的token流，并将其转发给SSE emitter
     */
    public static class StreamTokenConsumer implements TokenConsumerWithCompletion {
        private final SseEmitter emitter;
        private final AgentProcessor processor;
        private final EventService eventService;
        private final AtomicBoolean isCompleted = new AtomicBoolean(false);

        private Agent agent;
        private pangea.hiagent.web.dto.AgentRequest request;
        private String userId;
        private CompletionHandlerService completionHandlerService;
        private UserSseService userSseService;
        
        public StreamTokenConsumer(SseEmitter emitter, AgentProcessor processor, UserSseService unifiedSseService, EventService eventService, CompletionHandlerService completionHandlerService) {
            this.emitter = emitter;
            this.processor = processor;
            this.eventService = eventService;
            this.completionHandlerService = completionHandlerService;
            this.userSseService = unifiedSseService;
        }
        
        public void setContext(Agent agent, pangea.hiagent.web.dto.AgentRequest request, String userId) {
            this.agent = agent;
            this.request = request;
            this.userId = userId;
        }
        
        private boolean isEmitterValid() {
            if (userSseService != null && emitter != null) {
                return userSseService.isEmitterValidSafe(emitter);
            }
            return false;
        }

        @Override
        public void accept(String token) {
            // 使用JSON格式发送token，确保转义序列被正确处理
            try {
                if (!isCompleted.get()) {
                    // 检查emitter是否仍然有效
                    if (isEmitterValid()) {
                        // 检查是否是错误消息（以[错误]或[ERROR]开头）
                        if (token != null && (token.startsWith("[错误]") || token.startsWith("[ERROR]"))) {
                            // 发送标准错误事件而不是纯文本
                            eventService.sendErrorEvent(emitter, token);
                        } else {
                            // 使用SSE标准事件格式发送token，以JSON格式确保转义序列正确处理
                            eventService.sendTokenEvent(emitter, token);
                        }
                    } else {
                        log.debug("SSE emitter已无效，跳过发送token");
                    }
                }
            } catch (Exception e) {
                log.error("发送token失败", e);
            }
        }

        @Override
        public void onComplete(String fullContent) {
            // 处理完成时的回调
            if (isCompleted.getAndSet(true)) {
                log.debug("{} Agent处理已完成，跳过重复的完成回调", processor.getProcessorType());
                return;
            }

            log.info("{} Agent处理完成，总字符数: {}", processor.getProcessorType(), fullContent != null ? fullContent.length() : 0);

            try {
                // 检查emitter是否仍然有效，避免在连接已关闭时尝试处理完成事件
                if (userSseService != null && userSseService.isEmitterCompleted(emitter)) {
                    log.debug("SSE emitter已完成，跳过完成处理");
                    return;
                }
                
                // 使用CompletionHandlerService处理完成回调
                if (completionHandlerService != null) {
                    // 添加对CompletionHandlerService调用的额外保护，防止NoClassDefFoundError
                    try {
                        completionHandlerService.handleCompletion(emitter, processor, agent, request, userId, fullContent, isCompleted);
                    } catch (NoClassDefFoundError e) {
                        log.error("CompletionHandlerService依赖类未找到，使用默认处理逻辑: {}", e.getMessage());
                        // 如果类未找到，使用默认逻辑完成emitter
                        sendCompletionAndCompleteEmitter();
                    }
                } else {
                    // 如果completionHandlerService不可用，使用默认处理逻辑
                    sendCompletionAndCompleteEmitter();
                }
            } catch (Exception e) {
                log.error("处理完成事件失败", e);
                // 确保即使出现异常也完成emitter
                try {
                    sendCompletionAndCompleteEmitter();
                } catch (Exception ex) {
                    log.error("在异常处理路径中完成emitter也失败", ex);
                    // 最终保障：直接完成emitter，避免连接未关闭
                    try {
                        if (!userSseService.isEmitterCompleted(emitter)) {
                            emitter.complete();
                        }
                    } catch (Exception finalEx) {
                        log.error("最终尝试完成emitter也失败", finalEx);
                    }
                }
            }
        }
        
        /**
         * 安全地发送完成事件并完成emitter
         * 严格按照正确的关闭顺序：
         * 1. 完成所有业务处理（此处无业务处理）
         * 2. 发送完成信号
         * 3. 关闭SSE连接
         * 避免重复完成和异常情况
         * 注意：此方法不调用onComplete，避免循环调用
         */
        private void sendCompletionAndCompleteEmitter() {
            try {
                // 检查emitter是否已经完成，避免向已完成的连接发送数据
                if (userSseService != null && userSseService.isEmitterCompleted(emitter)) {
                    log.debug("SSE emitter已完成，跳过发送完成事件");
                    return;
                }
                
                // 发送完成事件
                emitter.send(SseEmitter.event().name("done").data("[DONE]").build());
                
                log.debug("完成信号已发送，准备关闭SSE连接");
                
                // 完成 emitter - 直接完成，不再通过CompletionHandlerService重复调用onComplete
                try {
                    if (userSseService != null && !userSseService.isEmitterCompleted(emitter)) {
                        emitter.complete();
                        log.debug("SSE连接已关闭");
                    }
                } catch (Exception ex) {
                    log.error("完成emitter时发生错误", ex);
                }
            } catch (Exception e) {
                log.error("发送完成事件时发生错误", e);
                
                // 尝试直接完成emitter
                try {
                    if (userSseService != null && !userSseService.isEmitterCompleted(emitter)) {
                        emitter.complete();
                        log.debug("异常路径下SSE连接已关闭");
                    }
                } catch (Exception ex) {
                    log.error("完成emitter时也发生错误", ex);
                }
            }
        }    }
}