package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import jakarta.annotation.PostConstruct;

import java.io.File;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * 动态模型加载器
 * 支持从外部JAR文件动态加载模型适配器
 */
@Slf4j
@Component
public class DynamicModelLoader {
    
    private static final String MODEL_ADAPTERS_DIR = "model-adapters";
    
    @PostConstruct
    public void init() {
        loadExternalModelAdapters();
    }
    
    /**
     * 加载外部模型适配器
     */
    public void loadExternalModelAdapters() {
        File adaptersDir = new File(MODEL_ADAPTERS_DIR);
        if (!adaptersDir.exists()) {
            log.info("模型适配器目录不存在: {}", MODEL_ADAPTERS_DIR);
            return;
        }
        
        if (!adaptersDir.isDirectory()) {
            log.warn("模型适配器路径不是目录: {}", MODEL_ADAPTERS_DIR);
            return;
        }
        
        File[] jarFiles = adaptersDir.listFiles((dir, name) -> name.endsWith(".jar"));
        if (jarFiles == null || jarFiles.length == 0) {
            log.info("模型适配器目录中没有找到JAR文件: {}", MODEL_ADAPTERS_DIR);
            return;
        }
        
        log.info("发现 {} 个模型适配器JAR文件", jarFiles.length);
        
        for (File jarFile : jarFiles) {
            try {
                loadModelAdaptersFromJar(jarFile.getAbsolutePath());
            } catch (Exception e) {
                log.error("加载模型适配器JAR文件失败: {}", jarFile.getName(), e);
            }
        }
    }
    
    /**
     * 从JAR文件加载模型适配器
     * 
     * @param jarPath JAR文件路径
     */
    public void loadModelAdaptersFromJar(String jarPath) {
        log.info("正在加载模型适配器JAR文件: {}", jarPath);
        
        // 使用try-with-resources确保资源正确关闭
        try (JarFile jarFile = new JarFile(jarPath);
             URLClassLoader classLoader = new URLClassLoader(
                     new URL[]{new File(jarPath).toURI().toURL()}, 
                     this.getClass().getClassLoader())) {
            
            // 扫描JAR文件中的类
            List<Class<?>> adapterClasses = new ArrayList<>();
            
            // 遍历JAR文件中的所有条目
            Enumeration<JarEntry> entries = jarFile.entries();
            while (entries.hasMoreElements()) {
                JarEntry entry = entries.nextElement();
                if (entry.getName().endsWith(".class")) {
                    try {
                        String className = entry.getName()
                                .replace('/', '.')
                                .replace('\\', '.')  // 处理Windows路径分隔符
                                .replace(".class", "");
                        
                        Class<?> clazz = classLoader.loadClass(className);
                        
                        // 检查类是否实现了ModelAdapter接口
                        if (ModelAdapter.class.isAssignableFrom(clazz) && 
                            !clazz.isInterface() && 
                            !clazz.equals(ModelAdapter.class)) {
                            adapterClasses.add(clazz);
                            log.info("发现模型适配器类: {}", className);
                        }
                    } catch (ClassNotFoundException e) {
                        log.warn("无法加载类: {}", entry.getName(), e);
                    } catch (NoClassDefFoundError e) {
                        log.warn("类定义缺失: {}", entry.getName(), e);
                    }
                }
            }
            
            // 注册发现的适配器
            registerModelAdapters(adapterClasses);
            
            log.info("成功从JAR文件加载 {} 个模型适配器", adapterClasses.size());
        } catch (Exception e) {
            log.error("从JAR文件加载模型适配器时发生错误: {}", jarPath, e);
            throw new RuntimeException("加载模型适配器失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 注册模型适配器
     * 
     * @param adapterClasses 适配器类列表
     */
    public void registerModelAdapters(List<Class<?>> adapterClasses) {
        for (Class<?> adapterClass : adapterClasses) {
            try {
                // 创建适配器实例
                Object instance = adapterClass.getDeclaredConstructor().newInstance();
                ModelAdapter adapter = (ModelAdapter) instance;
                
                // 注册适配器到ModelAdapterManager
                // 注意：这需要ModelAdapterManager支持动态注册
                log.info("成功注册模型适配器: {}", adapter.getProviderName());
            } catch (Exception e) {
                log.error("注册模型适配器失败: {}", adapterClass.getName(), e);
            } catch (NoClassDefFoundError e) {
                log.error("注册模型适配器失败，类定义缺失: {}", adapterClass.getName(), e);
            }
        }
    }
}