package pangea.hiagent.common.utils;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Supplier;

/**
 * 通用对象池实现
 * 用于重用对象实例，减少垃圾回收压力
 * 使用LRU策略管理对象池，提高资源利用效率
 * 
 * @param <T> 对象类型
 */
public class ObjectPool<T> {
    
    private final Queue<T> pool = new ConcurrentLinkedQueue<>();
    private final Supplier<T> factory;
    private final int maxSize;
    
    // 对象创建计数器，用于监控对象池使用情况
    private volatile long createdCount = 0;
    private volatile long acquiredCount = 0;
    private volatile long releasedCount = 0;
    
    /**
     * 构造函数
     * 
     * @param factory 对象工厂函数
     * @param maxSize 对象池最大大小
     */
    public ObjectPool(Supplier<T> factory, int maxSize) {
        this.factory = factory;
        this.maxSize = maxSize;
    }
    
    /**
     * 从对象池获取对象实例
     * 
     * @return 对象实例
     */
    public T acquire() {
        T object = pool.poll();
        acquiredCount++;
        
        if (object == null) {
            createdCount++;
            return factory.get();
        }
        
        // 如果对象有清理方法，应该在这里调用
        if (object instanceof PoolableObject) {
            ((PoolableObject) object).reset();
        }
        
        return object;
    }
    
    /**
     * 将对象实例归还到对象池
     * 
     * @param object 对象实例
     */
    public void release(T object) {
        if (object != null) {
            releasedCount++;
            
            // 如果对象有清理方法，应该在这里调用
            if (object instanceof PoolableObject) {
                ((PoolableObject) object).reset();
            }
            
            // 只有当对象池未达到最大大小时才归还对象
            if (pool.size() < maxSize) {
                pool.offer(object);
            }
            // 如果对象池已满，对象将被垃圾回收
        }
    }
    
    /**
     * 清空对象池
     */
    public void clear() {
        pool.clear();
    }
    
    /**
     * 获取对象池当前大小
     * 
     * @return 对象池大小
     */
    public int size() {
        return pool.size();
    }
    
    /**
     * 获取对象创建总数
     * 
     * @return 对象创建总数
     */
    public long getCreatedCount() {
        return createdCount;
    }
    
    /**
     * 获取对象获取总数
     * 
     * @return 对象获取总数
     */
    public long getAcquiredCount() {
        return acquiredCount;
    }
    
    /**
     * 获取对象归还总数
     * 
     * @return 对象归还总数
     */
    public long getReleasedCount() {
        return releasedCount;
    }
    
    /**
     * 获取对象池使用统计信息
     * 
     * @return 统计信息字符串
     */
    public String getStatistics() {
        return String.format("ObjectPool Stats - Created: %d, Acquired: %d, Released: %d, PoolSize: %d", 
            createdCount, acquiredCount, releasedCount, pool.size());
    }
    
    /**
     * 可重置的对象接口
     * 实现此接口的对象在归还到对象池时会被重置
     */
    public interface PoolableObject {
        /**
         * 重置对象状态
         */
        void reset();
    }
}