package pangea.hiagent.workpanel.event;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.web.dto.ToolEvent;
import pangea.hiagent.web.dto.WorkPanelEvent;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

/**
 * SSE事件广播器
 * 专门负责广播事件给所有订阅者
 */
@Slf4j
@Component
public class SseEventBroadcaster {
    
    @Autowired
    private UserSseService unifiedSseService;
    
    @Autowired
    private EventService eventService;
    
    /**
     * 广播工作面板事件给所有订阅者
     * 
     * @param event 工作面板事件
     */
    public void broadcastWorkPanelEvent(WorkPanelEvent event) {
        if (event == null) {
            log.warn("广播事件时接收到null事件");
            return;
        }
        
        try {
            // 预构建事件数据，避免重复构建
            Map<String, Object> eventData = eventService.buildWorkPanelEventData(event);
            
            try {
                // 获取所有emitter并广播
                List<SseEmitter> emitters = unifiedSseService.getEmitters();
                int successCount = 0;
                int failureCount = 0;
                
                // 使用CopyOnWriteArrayList避免并发修改异常
                for (SseEmitter emitter : new CopyOnWriteArrayList<>(emitters)) {
                    try {
                        // 检查emitter是否仍然有效
                        if (unifiedSseService.isEmitterValid(emitter)) {
                            emitter.send(SseEmitter.event().name("message").data(eventData));
                            successCount++;
                        } else {
                            // 移除无效的emitter
                            log.debug("移除无效的SSE连接");
                            unifiedSseService.removeEmitter(emitter);
                            failureCount++;
                        }
                    } catch (IOException e) {
                        log.error("发送事件失败，移除失效连接: {}", e.getMessage());
                        unifiedSseService.removeEmitter(emitter);
                        failureCount++;
                    } catch (IllegalStateException e) {
                        log.debug("Emitter已关闭，移除连接: {}", e.getMessage());
                        unifiedSseService.removeEmitter(emitter);
                        failureCount++;
                    } catch (Exception e) {
                        log.error("发送事件时发生未知异常，移除连接: {}", e.getMessage(), e);
                        unifiedSseService.removeEmitter(emitter);
                        failureCount++;
                    }
                }
                
                if (failureCount > 0) {
                    log.warn("事件广播部分失败: 成功={}, 失败={}", successCount, failureCount);
                }
                
                // 记录对象池使用统计信息（每100次广播记录一次）
                if ((successCount + failureCount) % 100 == 0) {
                    log.debug("对象池使用统计: {}", eventService.getMapPoolStatistics());
                }
            } finally {
                // 确保eventData被归还到对象池
                eventService.releaseMap(eventData);
            }
        } catch (Exception e) {
            String toolName = null;
            if (event instanceof ToolEvent) {
                toolName = ((ToolEvent) event).getToolName();
            }
            log.error("广播事件失败: 事件类型={}, 工具={}, 错误信息={}", 
                    event.getType(), 
                    toolName, 
                    e.getMessage(), 
                    e);
        }
    }
}