package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;

import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.processor.AgentProcessorFactory;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.web.dto.ChatRequest;
import pangea.hiagent.model.Agent;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.web.dto.AgentRequest;
import pangea.hiagent.workpanel.event.EventService;
import jakarta.servlet.http.HttpServletResponse;

/**
 * Agent 对话服务
 * 职责：协调整个AI对话流程，作为流式处理的统一入口和协调者
 */
@Slf4j
@Service
public class AgentChatService {

    private final ErrorHandlerService errorHandlerService;
    private final AgentProcessorFactory agentProcessorFactory;
    private final AgentToolManager agentToolManager;
    private final UserSseService userSseService;
    private final pangea.hiagent.web.service.AgentService agentService;
    private final SseTokenEmitter sseTokenEmitter;

    public AgentChatService(
            EventService eventService,
            ErrorHandlerService errorHandlerService,
            AgentProcessorFactory agentProcessorFactory,
            AgentToolManager agentToolManager,
            UserSseService userSseService,
            pangea.hiagent.web.service.AgentService agentService,
            SseTokenEmitter sseTokenEmitter) {
        this.errorHandlerService = errorHandlerService;
        this.agentProcessorFactory = agentProcessorFactory;
        this.agentToolManager = agentToolManager;
        this.userSseService = userSseService;
        this.agentService = agentService;
        this.sseTokenEmitter = sseTokenEmitter;
    }

    // /**
    // * 处理同步对话请求的统一入口
    // * @param agent Agent对象
    // * @param request 请求对象
    // * @param userId 用户ID
    // * @return 处理结果
    // */
    // public String handleChatSync(Agent agent, AgentRequest request, String
    // userId) {
    // log.info("开始处理同步对话请求，AgentId: {}, 用户消息: {}", agent.getId(),
    // request.getUserMessage());
    //
    // try {
    // // 获取处理器
    // AgentProcessor processor = agentProcessorFactory.getProcessor(agent);
    // if (processor == null) {
    // log.error("无法获取Agent处理器");
    // return "[错误] 无法获取Agent处理器";
    // }
    //
    // // 处理请求
    // return processor.processRequest(agent, request, userId);
    // } catch (Exception e) {
    // log.error("处理普通Agent请求时发生错误", e);
    // return "[错误] 处理请求时发生错误: " + e.getMessage();
    // }
    // }

    /**
     * 处理流式对话请求的统一入口
     * 
     * @param agentId     Agent ID
     * @param chatRequest 对话请求
     * @param response    HTTP响应
     * @return SSE emitter
     */
    public SseEmitter handleChatStream(String agentId, ChatRequest chatRequest, HttpServletResponse response) {
        log.info("开始处理流式对话请求，AgentId: {}, 用户消息: {}", agentId, chatRequest.getMessage());

        // 尝试获取当前用户ID，优先从SecurityContext获取，其次从请求中解析JWT
        String userId = UserUtils.getCurrentUserId();

        // 如果在主线程中未能获取到用户ID，尝试在异步环境中获取
        if (userId == null) {
            userId = UserUtils.getCurrentUserIdInAsync();
        }

        if (userId == null) {
            log.error("用户未认证");
            SseEmitter emitter = userSseService.createEmitter();
            // 检查响应是否已经提交
            if (!response.isCommitted()) {
                errorHandlerService.handleChatError(emitter, "用户未认证，请重新登录");
            } else {
                log.warn("响应已提交，无法发送用户未认证错误信息");
                // 检查emitter是否已经完成，避免重复关闭
                if (!userSseService.isEmitterCompleted(emitter)) {
                    emitter.complete();
                }
            }
            return emitter;
        }

        // 验证Agent是否存在
        Agent agent = agentService.getAgent(agentId);
        if (agent == null) {
            log.warn("Agent不存在: {}", agentId);
            SseEmitter emitter = userSseService.createEmitter();
            // 检查响应是否已经提交
            if (!response.isCommitted()) {
                errorHandlerService.handleChatError(emitter, "Agent不存在");
            } else {
                log.warn("响应已提交，无法发送Agent不存在错误信息");
                // 检查emitter是否已经完成，避免重复关闭
                if (!userSseService.isEmitterCompleted(emitter)) {
                    emitter.complete();
                }
            }
            return emitter;
        }

        // 创建 SSE emitter
        SseEmitter emitter = userSseService.createEmitter();

        // 异步处理对话，避免阻塞HTTP连接
        processChatStreamAsync(emitter, agent, chatRequest, userId);

        return emitter;
    }

    /**
     * 异步处理流式对话
     */
    @Async
    private void processChatStreamAsync(SseEmitter emitter, Agent agent, ChatRequest chatRequest, String userId) {
        try {
            processChatRequest(emitter, agent, chatRequest, userId);
        } catch (Exception e) {
            log.error("处理聊天请求时发生异常", e);

            // 检查响应是否已经提交
            if (emitter != null && !userSseService.isEmitterCompleted(emitter)) {
                errorHandlerService.handleChatError(emitter, "处理请求时发生错误", e, null);
            }
        }
    }

    /**
     * 处理聊天请求的核心逻辑
     * 注意：权限验证已在主线程中完成，此正仅执行业务逻辑不进行权限检查
     *
     * @param emitter     SSE发射器
     * @param agent       Agent对象
     * @param chatRequest 聊天请求
     * @param userId      用户ID
     */
    private void processChatRequest(SseEmitter emitter, Agent agent, ChatRequest chatRequest, String userId) {
        try {
            // 参数验证
            if (!validateParameters(emitter, agent, chatRequest, userId)) {
                return;
            }

            // 获取处理器
            AgentProcessor processor = agentProcessorFactory.getProcessor(agent);
            if (processor == null) {
                log.error("无法获取Agent处理器，Agent: {}", agent.getId());
                errorHandlerService.handleChatError(emitter, "无法获取Agent处理器");
                return;
            }

            // 转换请求对象
            AgentRequest request = chatRequest.toAgentRequest(agent.getId(), agent, agentToolManager);

            // 创建新的SseTokenEmitter实例
            SseTokenEmitter tokenEmitter = sseTokenEmitter.createNewInstance(emitter, agent, request, userId, this::handleCompletion);
            
            // 处理流式请求
            processor.processStreamRequest(request, agent, userId, tokenEmitter);
        } catch (Exception e) {
            log.error("处理聊天请求时发生异常", e);
            errorHandlerService.handleChatError(emitter, "处理请求时发生错误", e, null);
        }
    }

    /**
     * 处理完成回调
     * 
     * @param emitter     SSE发射器
     * @param agent       Agent对象
     * @param request     Agent请求
     * @param userId      用户ID
     * @param fullContent 完整内容
     */
    private void handleCompletion(SseEmitter emitter, Agent agent, AgentRequest request, String userId,
            String fullContent) {
        log.info("Agent处理完成，总字符数: {}", fullContent != null ? fullContent.length() : 0);

        // 保存对话记录
        try {
            saveDialogue(agent, request, userId, fullContent);
            log.info("对话记录保存成功");
        } catch (Exception e) {
            log.error("保存对话记录失败", e);
            // 记录异常但不中断流程
        }
    }

    /**
     * 保存对话记录
     */
    private void saveDialogue(Agent agent, AgentRequest request, String userId, String responseContent) {
        // 参数验证
        if (agent == null || request == null || userId == null || userId.trim().isEmpty()) {
            log.error("保存对话记录失败：参数无效");
            return;
        }

        try {
            // 创建对话记录
            pangea.hiagent.model.AgentDialogue dialogue = pangea.hiagent.model.AgentDialogue.builder()
                    .agentId(request.getAgentId())
                    .userMessage(request.getUserMessage())
                    .agentResponse(responseContent)
                    .userId(userId)
                    .build();

            // 保存对话记录
            agentService.saveDialogue(dialogue);
        } catch (Exception e) {
            log.error("保存对话记录失败", e);
            throw new RuntimeException("保存对话记录失败", e);
        }
    }

    /**
     * 验证所有必需参数
     * 
     * @param emitter     SSE发射器
     * @param agent       Agent对象
     * @param chatRequest 聊天请求
     * @param userId      用户ID
     * @return 验证是否通过
     */
    private boolean validateParameters(SseEmitter emitter, Agent agent, ChatRequest chatRequest, String userId) {
        return emitter != null && chatRequest != null && agent != null && userId != null && !userId.isEmpty();
    }
}