package pangea.hiagent.tool.impl;

import com.microsoft.playwright.*;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.microsoft.playwright.options.LoadState;
import com.microsoft.playwright.options.WaitUntilState;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import jakarta.annotation.PreDestroy;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import pangea.hiagent.web.service.ToolConfigService;
import pangea.hiagent.workpanel.playwright.PlaywrightManager;

/**
 * 海信SSO认证工具类
 * 用于访问需要SSO认证的海信业务系统，自动完成登录并提取页面内容
 */
@Slf4j
@Component
public class HisenseSsoLoginTool {

    // SSO登录页面URL
    private static final String SSO_LOGIN_URL = "https://sso.hisense.com/login/";

    private static final String SSO_PROFILE_URL = "https://sso.hisense.com/selfcare/?#/profile";

    private static final String SSO_MFA_URL = "https://sso.hisense.com/login/mfaLogin.html";

    // 登录成功后可能跳转的URL模式
    private static final String[] SUCCESS_REDIRECT_URLS = {
        "https://sso.hisense.com/selfcare/?#/profile",
        "https://sso.hisense.com/selfcare/",
        "https://sso.hisense.com/login/success",
        "https://sso.hisense.com/dashboard"
    };

    // 等待URL超时时间（毫秒），从30秒增加到60秒
    private static final int WAIT_FOR_URL_TIMEOUT = 60000;

    // 等待URL超时时间（毫秒）用于MFA验证
    private static final int MFA_WAIT_FOR_URL_TIMEOUT = 45000;

    // 用户名输入框选择器
    private static final String USERNAME_INPUT_SELECTOR = "input[placeholder='账号名/海信邮箱/手机号']";

    // 密码输入框选择器
    private static final String PASSWORD_INPUT_SELECTOR = "input[placeholder='密码'][type='password']";

    // 登录按钮选择器
    private static final String LOGIN_BUTTON_SELECTOR = "#login-button";

    // 注入Playwright管理器
    @Autowired
    private PlaywrightManager playwrightManager;

    // 上次登录时间
    private long lastLoginTime = 0;

    @Autowired
    private ToolConfigService toolConfigService;

    // MFA 会话信息容器
    private static class MfaSession {
        Page page;
        BrowserContext context;
        long lastAccessTime;
        
        MfaSession(Page page, BrowserContext context) {
            this.page = page;
            this.context = context;
            this.lastAccessTime = System.currentTimeMillis();
        }
        
        void updateAccessTime() {
            this.lastAccessTime = System.currentTimeMillis();
        }
        
        boolean isExpired(long timeoutMillis) {
            return System.currentTimeMillis() - lastAccessTime > timeoutMillis;
        }
    }
    
    // MFA 会话缓存（用户名 -> MFA会话）
    private final ConcurrentMap<String, MfaSession> mfaSessions = new ConcurrentHashMap<>();
    
    // MFA 会话超时时间（15分钟），短于BrowserContext超时时间以主动清理
    private static final long MFA_SESSION_TIMEOUT = 15 * 60 * 1000;

    // 登录状态有效期（毫秒），设置为30分钟
    private static final long LOGIN_VALIDITY_PERIOD = 30 * 60 * 1000;

    private String userName;
    private String password;

    public String getUserName() {
        userName = toolConfigService.getParamValue("hisenseSsoLogin", "ssoUsername");
        return userName;
    }

    private String getPassword() {
        password = toolConfigService.getParamValue("hisenseSsoLogin", "ssoPassword");
        return password;
    }

    private BrowserContext getUSerContext() {
        return playwrightManager.getUserContext(getUserName());
    }

    /**
     * 销毁Playwright资源
     */
    @PreDestroy
    public void destroy() {
        try {
            // 清空MFA会话缓存
            mfaSessions.clear();
            log.info("海信SSO认证工具的MFA会话缓存已清空");
            
            // 注意：不在这里关闭BrowserContext，由PlaywrightManager统一管理生命周期
            // 避免在MFA验证进行中被意外关闭
        } catch (Exception e) {
            log.error("海信SSO认证工具的资源释放失败: ", e);
        }
    }
    
    /**
     * 获取和更新 MFA 会话
     * 同时检查会话是否过期，并在访问时更新最后访问时间
     * 
     * @param username 用户名
     * @return MFA会话，如果会话已过期或不存在则返回null
     */
    private MfaSession getMfaSessionAndUpdateTime(String username) {
        MfaSession session = mfaSessions.get(username);
        
        if (session == null) {
            return null;
        }
        
        // 检查会话是否过期
        if (session.isExpired(MFA_SESSION_TIMEOUT)) {
            log.warn("MFA会话已过期，用户: {}", username);
            mfaSessions.remove(username);
            return null;
        }
        
        // 更新最后访问时间（保活）
        session.updateAccessTime();
        return session;
    }
    
    /**
     * 清理过期的MFA会话
     */
    private void cleanupExpiredMfaSessions() {
        mfaSessions.entrySet().removeIf(entry -> {
            if (entry.getValue().isExpired(MFA_SESSION_TIMEOUT)) {
                log.info("清理过期的MFA会话: {}", entry.getKey());
                return true;
            }
            return false;
        });
    }

    /**
     * 工具方法：获取海信业务系统的网页内容（自动处理SSO认证）
     * 
     * @param businessSystemUrl 海信业务系统页面URL
     * @return 页面内容（HTML文本）
     */
    @Tool(description = "获取任意海信业务系统的网页内容（自动处理SSO认证）")
    public String getHisenseBusinessSystemContent(
            @JsonPropertyDescription("海信业务系统的页面URL") String businessSystemUrl) {
        // initializeIfNeeded();
        log.info("开始获取海信业务系统内容，URL: {}", businessSystemUrl);

        String ssoUsername = getUserName();
        String ssoPassword = getPassword();
        // 校验SSO凭证是否配置
        if (ssoUsername == null || ssoUsername.isEmpty() || ssoPassword == null || ssoPassword.isEmpty()) {
            String errorMsg = "SSO用户名或密码未配置，海信SSO工具不可用";
            log.warn(errorMsg);
            return errorMsg;
        }

        long startTime = System.currentTimeMillis();

        // 参数校验
        if (businessSystemUrl == null || businessSystemUrl.isEmpty()) {
            String errorMsg = "业务系统URL不能为空";
            log.error(errorMsg);
            return errorMsg;
        }

        Page page = null;

        try {
            // 检查是否已有有效的登录会话
            boolean sessionValid = isSessionLoggedIn() && validateSession(businessSystemUrl);

            if (sessionValid) {
                log.info("检测到有效会话，直接使用共享上下文");
                page = getUSerContext().newPage();
            } else {
                log.info("未检测到有效会话，使用共享上下文并重新登录");
                page = getUSerContext().newPage();

                // 访问业务系统页面
                log.info("正在访问业务系统页面: {}", businessSystemUrl);
                page.navigate(businessSystemUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));

                // 检查是否重定向到了SSO登录页面
                String currentUrl = page.url();
                log.info("当前页面URL: {}", currentUrl);

                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    log.info("检测到SSO登录页面，开始自动登录...");
                    // 执行SSO登录
                    performLoginAndUpdateStatus(page);

                    // 等待登录完成并重定向回业务系统
                    boolean redirected = waitForUrlWithMultipleOptions(page, new String[]{businessSystemUrl}, WAIT_FOR_URL_TIMEOUT);
                    if (!redirected) {
                        log.warn("未能在指定时间内重定向到业务系统页面，当前URL: {}", page.url());
                    } else {
                        log.info("登录成功，已重定向回业务系统页面");
                    }
                } else {
                    // 即使没有跳转到登录页面，也更新登录时间
                    lastLoginTime = System.currentTimeMillis();
                    log.info("直接访问业务系统页面成功，无需SSO登录，更新会话时间");
                }
            }

            // 如果页面尚未导航到业务系统URL，则导航到该URL
            if (!page.url().equals(businessSystemUrl) && !page.url().startsWith(businessSystemUrl)) {
                log.info("正在访问业务系统页面: {}", businessSystemUrl);
                page.navigate(businessSystemUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
            }

            // 提取页面内容
            String content = page.locator("body").innerText();
            long endTime = System.currentTimeMillis();
            log.info("成功获取业务系统页面内容，耗时: {} ms", endTime - startTime);

            // 检查是否包含错误信息
            if (content.contains("InvalidStateError") && content.contains("setRequestHeader")) {
                log.warn("检测到页面中可能存在JavaScript错误，但这不会影响主要功能");
            }

            return content;
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "获取海信业务系统内容失败: " + e.getMessage();
            log.error("获取海信业务系统内容失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        } finally {
            // 释放页面资源
            if (page != null) {
                try {
                    page.close();
                } catch (Exception e) {
                    log.warn("关闭页面时发生异常: {}", e.getMessage());
                }
            }
        }
    }

    /**
     * 工具方法：海信SSO登录工具，用于登录海信SSO系统
     * 
     * @param username 用户名
     * @param password 密码
     * @return 登录结果
     */
    @Tool(description = "海信SSO登录工具，用于登录海信SSO系统")
    public String hisenseSsoLogin() {
        String username = getUserName();
        String password = getPassword();
        // 校验SSO凭证是否配置
        if (username == null || username.isEmpty() || password == null || password.isEmpty()) {
            String errorMsg = "SSO用户名或密码未配置，海信SSO工具不可用";
            log.warn(errorMsg);
            return errorMsg;
        }
        log.info("开始执行海信SSO登录，用户名: {}", username);

        long startTime = System.currentTimeMillis();

        // 参数校验
        if (username == null || username.isEmpty()) {
            String errorMsg = "用户名不能为空";
            log.error(errorMsg);
            return errorMsg;
        }

        if (password == null || password.isEmpty()) {
            String errorMsg = "密码不能为空";
            log.error(errorMsg);
            return errorMsg;
        }

        Page page = null;

        try {
            // 访问SSO登录页面
            log.info("正在访问SSO登录页面: {}", SSO_LOGIN_URL);
            page = getUSerContext().newPage();
            page.navigate(SSO_LOGIN_URL, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));

            if (page.url().equals(SSO_MFA_URL)) {
                log.info("检测到MFA页面，自动发送验证码...");
                // 执行MFA登录，传递Context以保证生命周期管理
                return sendVerificationCode(username, page, getUSerContext());
            }

            if (!SSO_LOGIN_URL.equals(page.url())) {
                return "海信SSO在之前已登录成功";
            }

            // 执行SSO登录
            String loginResult = performLoginAndUpdateStatus(page);

            if(!loginResult.equals("SSO登录成功")){
                return loginResult;
            }

            // 等待登录完成并重定向到SSO配置页面，使用更灵活的URL匹配和更长的超时时间
            boolean profileRedirected = waitForSpecificUrl(page, SSO_PROFILE_URL, WAIT_FOR_URL_TIMEOUT);
            if (!profileRedirected) {
                // 如果没有跳转到预期的配置页面，检查是否跳转到了其他可能的登录成功页面
                boolean alternativeRedirected = waitForUrlWithMultipleOptions(page, SUCCESS_REDIRECT_URLS, WAIT_FOR_URL_TIMEOUT);
                if (!alternativeRedirected) {
                    log.warn("未能在指定时间内重定向到SSO配置页面，当前URL: {}", page.url());
                } else {
                    log.info("登录成功，已重定向到SSO相关页面");
                }
            } else {
                log.info("登录成功，已重定向回SSO配置页面");
            }

            long endTime = System.currentTimeMillis();
            log.info("海信SSO登录完成，耗时: {} ms", endTime - startTime);

            return "海信SSO登录成功";
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "海信SSO登录失败: " + e.getMessage();
            log.error("海信SSO登录失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        } finally {
            // 释放页面资源
            if (page != null && !page.url().equals(SSO_MFA_URL)) {
                try {
                    page.close();
                } catch (Exception e) {
                    log.warn("关闭页面时发生异常: {}", e.getMessage());
                }
            }
        }
    }

    /**
     * 等待页面跳转到指定的多个URL选项中的任意一个
     * 
     * @param page 要监控的页面
     * @param urls 可能的目标URL数组
     * @param timeout 超时时间（毫秒）
     * @return 是否成功跳转到目标URL之一
     */
    private boolean waitForUrlWithMultipleOptions(Page page, String[] urls, int timeout) {
        long startTime = System.currentTimeMillis();
        long endTime = startTime + timeout;
        
        // 首先检查当前URL是否已经是目标URL之一
        String currentUrl = page.url();
        for (String url : urls) {
            if (currentUrl.equals(url) || currentUrl.startsWith(url)) {
                log.info("当前URL已经匹配目标URL: {}", currentUrl);
                return true;
            }
        }
        
        // 监控URL变化直到超时或匹配到目标URL
        while (System.currentTimeMillis() < endTime) {
            try {
                // 等待URL变化 - 使用URL匹配器和超时选项
                page.waitForURL(url -> {
                    // 检查新URL是否匹配目标URL之一
                    for (String targetUrl : urls) {
                        if (url.equals(targetUrl) || url.startsWith(targetUrl)) {
                            log.info("成功跳转到目标URL: {}", url);
                            return true;
                        }
                    }
                    return false;
                }, new Page.WaitForURLOptions().setTimeout(1000)); // 短超时，用于检测变化
                
                // 如果上面的waitForURL成功，说明已经匹配到目标URL
                return true;
                
            } catch (com.microsoft.playwright.TimeoutError e) {
                // 1秒内URL未变化，继续轮询
                String current = page.url();
                for (String url : urls) {
                    if (current.equals(url) || current.startsWith(url)) {
                        log.info("检测到目标URL: {}", current);
                        return true;
                    }
                }
            } catch (Exception e) {
                // 记录异常但继续轮询，因为可能是临时错误
                log.debug("等待URL变化时发生异常: {}", e.getMessage());
            }
            
            // 短暂休眠以避免过度占用CPU
            try {
                Thread.sleep(500);
            } catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                log.warn("等待URL过程中被中断");
                return false;
            }
        }
        
        log.warn("等待URL超时，当前URL: {}，期望URLs: {}", page.url(), String.join(",", urls));
        return false;
    }

    /**
     * 等待页面跳转到指定URL，使用更灵活的匹配方式
     * 
     * @param page 要监控的页面
     * @param expectedUrl 期望的目标URL
     * @param timeout 超时时间（毫秒）
     * @return 是否成功跳转到目标URL
     */
    private boolean waitForSpecificUrl(Page page, String expectedUrl, int timeout) {
        long startTime = System.currentTimeMillis();
        long endTime = startTime + timeout;
        
        while (System.currentTimeMillis() < endTime) {
            try {
                String currentUrl = page.url();
                
                // 检查当前URL是否匹配
                if (currentUrl.equals(expectedUrl) || currentUrl.startsWith(expectedUrl)) {
                    log.info("已匹配到目标URL: {}", currentUrl);
                    return true;
                }
                
                // 等待URL变化到期望的URL
                page.waitForURL(url -> url.equals(expectedUrl) || url.startsWith(expectedUrl), 
                    new Page.WaitForURLOptions().setTimeout(1000));
                
                // 如果上面的waitForURL成功，说明已经匹配到目标URL
                return true;
                
            } catch (com.microsoft.playwright.TimeoutError e) {
                // 1秒内URL未变化，继续轮询
                String current = page.url();
                if (current.equals(expectedUrl) || current.startsWith(expectedUrl)) {
                    log.info("检测到目标URL: {}", current);
                    return true;
                }
            } catch (Exception e) {
                // 检查当前URL是否匹配，因为异常可能表示页面状态变化
                try {
                    String current = page.url();
                    if (current.equals(expectedUrl) || current.startsWith(expectedUrl)) {
                        log.info("异常后检测到目标URL: {}", current);
                        return true;
                    }
                } catch (Exception urlException) {
                    log.debug("获取当前URL时发生异常: {}", urlException.getMessage());
                }
                
                // 如果是TargetClosedError，直接返回失败
                if (e instanceof com.microsoft.playwright.impl.TargetClosedError) {
                    log.error("页面或浏览器上下文已关闭: {}", e.getMessage());
                    return false;
                }
            }
            
            // 短暂休眠
            try {
                Thread.sleep(500);
            } catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                log.warn("等待URL过程中被中断");
                return false;
            }
        }
        
        log.warn("等待特定URL超时，当前URL: {}，期望URL: {}", page.url(), expectedUrl);
        return false;
    }

    private String sendVerificationCode(String username, Page page, BrowserContext context) {
        try {
            // 最初检查页面有效性
            if (!isPageValid(page)) {
                String errorMsg = "发送验证码时页面已关闭";
                log.error(errorMsg);
                return errorMsg;
            }

            // 确定SSO_MFA_URL网页加载完成
            try {
                page.waitForLoadState(LoadState.NETWORKIDLE);
            } catch (Exception e) {
                log.warn("等待页面加载时发生异常（可能页面已关闭），尝试继续: {}", e.getMessage());
            }

            // 点击获取验证码按钮
            if (!isPageValid(page)) {
                String errorMsg = "在操作前页面已关闭";
                log.error(errorMsg);
                return errorMsg;
            }

            Locator getCodeButton = page.locator("div.get-code-btn[hk-ripple][hk-ripple-color]");
            if (getCodeButton.count() > 0) {
                getCodeButton.click();
                log.info("已点击获取验证码按钮");
            } else {
                String errorMsg = "未找到获取验证码按钮";
                log.warn(errorMsg);
                throw new RuntimeException(errorMsg);
            }

            // 确定页面上出现提示文字
            Locator smsSentTip = page.locator("#sms-sent-tip");
            try {
                page.waitForSelector("#sms-sent-tip", new Page.WaitForSelectorOptions()
                        .setState(com.microsoft.playwright.options.WaitForSelectorState.VISIBLE).setTimeout(10000));
                String tipText = smsSentTip.textContent();
                if (tipText != null && tipText.contains("短信验证码已发送")) {
                    log.info("验证码发送成功: {}", tipText);

                    // 保存 MFA 会话（包括 Page 和 Context），便于后续验证使用
                    MfaSession session = new MfaSession(page, context);
                    mfaSessions.put(username, session);
                    
                    return "已发送验证码，请查看短信";
                } else {
                    String errorMsg = "验证码发送提示信息不符合预期: " + tipText;
                    log.warn(errorMsg);
                    throw new RuntimeException(errorMsg);
                }
            } catch (Exception e) {
                log.error("等待验证码发送提示失败: ", e);
                throw new RuntimeException("未能确认验证码发送成功: " + e.getMessage());
            }
        } catch (com.microsoft.playwright.impl.TargetClosedError e) {
            // 专门处理TargetClosedError
            log.error("验证码发送过程中BrowserContext已关闭，错误: {}", e.getMessage());
            mfaSessions.remove(username);
            return "验证码发送失败：BrowserContext已关闭，请重试";
        } catch (Exception e) {
            log.error("验证码发送过程发生异常，错误类型: {}，详情: ", e.getClass().getName(), e);
            mfaSessions.remove(username);
            throw e;
        }
    }

    /**
     * 检查页面是否仍然有效
     * 
     * @param page 要检查的页面
     * @return 如果页面有效则返回true
     */
    private boolean isPageValid(Page page) {
        if (page == null) {
            return false;
        }
        try {
            // 尝试访问页面属性来检查它是否仍然有效
            page.url();
            return true;
        } catch (Exception e) {
            log.debug("页面已关闭或无效: {}", e.getMessage());
            return false;
        }
    }
    
    /**
     * 检查BrowserContext是否仍然有效
     * 
     * @param context 要检查的BrowserContext
     * @return 如果Context有效则返回true
     */
    private boolean isContextValid(BrowserContext context) {
        if (context == null) {
            return false;
        }
        try {
            // 尝试访问Context属性来检查它是否仍然有效
            context.pages();
            return true;
        } catch (Exception e) {
            log.debug("BrowserContext已关闭或无效: {}", e.getMessage());
            return false;
        }
    }

    /**
     * 工具方法：处理MFA验证码验证，完成海信SSO登录
     * 
     * @param verificationCode 验证码
     * @return 验证结果
     */
    @Tool(description = "处理MFA验证码验证，完成海信SSO登录")
    public String handleMfaVerification(
            @JsonPropertyDescription("短信验证码") String verificationCode) {
        log.info("开始处理MFA验证码验证");

        String username = getUserName();

        // 参数校验
        if (verificationCode == null || verificationCode.isEmpty()) {
            String errorMsg = "验证码不能为空";
            log.error(errorMsg);
            return errorMsg;
        }

        long startTime = System.currentTimeMillis();
        
        // 清理过期的MFA会话
        cleanupExpiredMfaSessions();

        // 获取当前用户的MFA会话并更新访问时间
        MfaSession mfaSession = getMfaSessionAndUpdateTime(username);
        if (mfaSession == null) {
            String errorMsg = "未找到当前用户的MFA验证会话，请先触发验证码发送。如果30分钟内未使用验证码，会话将自动过期。";
            log.error(errorMsg);
            return errorMsg;
        }
        
        Page mfaPage = mfaSession.page;
        BrowserContext context = mfaSession.context;

        // 检查MFA页面和Context是否仍然有效，如果已关闭则返回错误
        if (!isPageValid(mfaPage) || !isContextValid(context)) {
            String errorMsg = "MFA验证页面或Context已关闭，BrowserContext可能已被释放，请重新触发验证码发送流程";
            log.error(errorMsg);
            mfaSessions.remove(username);
            return errorMsg;
        }

        try {
            // 等待页面加载完成，添加异常处理
            try {
                mfaPage.waitForLoadState(LoadState.NETWORKIDLE);
            } catch (Exception e) {
                log.warn("等待页面加载时发生异常（可能页面已关闭），尝试继续: {}", e.getMessage());
            }

            // 查找验证码输入框并填入验证码
            // 在访问元素前再次检查页面有效性
            if (!isPageValid(mfaPage) || !isContextValid(context)) {
                String errorMsg = "MFA验证页面或Context在操作过程中被关闭";
                log.error(errorMsg);
                mfaSessions.remove(username);
                return errorMsg;
            }

            Locator verificationInput = mfaPage.locator("input[placeholder='请输入短信验证码'][name='']");
            if (verificationInput.count() == 0) {
                String errorMsg = "未找到验证码输入框";
                log.error(errorMsg);
                return errorMsg;
            }
            verificationInput.fill(verificationCode);
            log.info("验证码已填入输入框");

            // 再次检查页面有效性
            if (!isPageValid(mfaPage) || !isContextValid(context)) {
                String errorMsg = "验证码填入后页面或Context已关闭";
                log.error(errorMsg);
                mfaSessions.remove(username);
                return errorMsg;
            }

            // 点击登录按钮提交验证码
            Locator loginButton = mfaPage.locator("button#login-button.para-btn.para-btn-login[hk-ripple='']");
            if (loginButton.count() == 0) {
                String errorMsg = "未找到登录按钮";
                log.error(errorMsg);
                return errorMsg;
            }
            loginButton.click();
            log.info("已点击登录按钮提交验证码");

            // 等待页面跳转，确认登录结果
            try {
                // 等待页面离开MFA页面，使用轮询方式检查URL变化
                boolean pageLeftMfa = false;
                long mfaStartTime = System.currentTimeMillis();
                long mfaEndTime = mfaStartTime + MFA_WAIT_FOR_URL_TIMEOUT;
                
                while (System.currentTimeMillis() < mfaEndTime) {
                    String currentUrl = mfaPage.url();
                    if (!currentUrl.equals(SSO_MFA_URL)) {
                        pageLeftMfa = true;
                        log.info("MFA验证成功，已跳转到: {}", currentUrl);
                        break;
                    }
                    
                    // 短暂休眠后继续检查
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        log.warn("MFA等待过程中被中断");
                        break;
                    }
                }
                
                if (pageLeftMfa) {
                    // 从缓存中移除会话，因为登录已完成
                    mfaSessions.remove(username);

                    // 更新登录时间
                    lastLoginTime = System.currentTimeMillis();

                    long endTime = System.currentTimeMillis();
                    log.info("MFA验证完成，耗时: {} ms", endTime - startTime);

                    return "MFA验证成功，登录完成";
                } else {
                    // 如果仍在MFA页面，说明可能超时但验证仍在进行中，也认为成功
                    log.info("MFA验证可能仍在进行中，当前仍在MFA页面");
                    
                    // 从缓存中移除会话
                    mfaSessions.remove(username);

                    // 更新登录时间
                    lastLoginTime = System.currentTimeMillis();

                    long endTime = System.currentTimeMillis();
                    log.info("MFA验证处理完成，耗时: {} ms", endTime - startTime);

                    return "MFA验证已处理";
                }
            } catch (Exception urlException) {
                // 检查是否仍然是MFA页面，表示登录失败
                String currentUrl = mfaPage.url();
                if (currentUrl.equals(SSO_MFA_URL)) {
                    String errorMsg = "MFA验证失败，验证码可能错误或已过期，请重试";
                    log.error(errorMsg);
                    return errorMsg;
                } else {
                    // 页面已跳转，说明登录成功
                    log.info("MFA验证成功，已跳转到: {}", currentUrl);

                    // 从缓存中移除会话
                    mfaSessions.remove(username);

                    // 更新登录时间
                    lastLoginTime = System.currentTimeMillis();

                    long endTime = System.currentTimeMillis();
                    log.info("MFA验证完成，耗时: {} ms", endTime - startTime);

                    return "MFA验证成功，登录完成";
                }
            }
        } catch (com.microsoft.playwright.impl.TargetClosedError e) {
            // 专门处理TargetClosedError
            long endTime = System.currentTimeMillis();
            String errorMsg = "MFA验证时BrowserContext已关闭，请重新触发验证码发送流程";
            log.error("MFA验证失败 - TargetClosedError，耗时: {} ms，完整错误堆栈: ", endTime - startTime, e);
            mfaSessions.remove(username);
            return errorMsg;
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "MFA验证过程发生异常: " + e.getMessage();
            log.error("MFA验证失败，耗时: {} ms，错误类型: {}，完整错误堆栈: ", endTime - startTime, e.getClass().getName(), e);
            mfaSessions.remove(username);
            return errorMsg;
        }
    }

    /**
     * 工具方法：海信SSO登出工具，用于退出海信SSO系统
     * 
     * @return 登出结果
     */
    @Tool(description = "海信SSO登出工具，用于退出海信SSO系统")
    public String hisenseSsoLogout() {
        // initializeIfNeeded();
        log.info("开始执行海信SSO登出");

        long startTime = System.currentTimeMillis();

        try {
            // 关闭共享上下文
            if (getUSerContext() != null) {
                getUSerContext().close();
                log.info("共享上下文已关闭");
            }

            // 重置登录时间
            lastLoginTime = 0;
            log.info("登录时间已重置");

            long endTime = System.currentTimeMillis();
            log.info("海信SSO登出完成，耗时: {} ms", endTime - startTime);

            return "海信SSO登出成功";
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "海信SSO登出失败: " + e.getMessage();
            log.error("海信SSO登出失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        }
    }

    /**
     * 工具方法：检查海信SSO登录状态
     */
    @Tool(description = "检查海信SSO登录状态")
    public String checkHisenseSsoLoginStatus() {
        // initializeIfNeeded();
        log.info("开始检查海信SSO登录状态");

        long startTime = System.currentTimeMillis();

        try {
            boolean isLoggedIn = isSessionLoggedIn();
            long endTime = System.currentTimeMillis();
            log.info("海信SSO登录状态检查完成:{}，耗时: {} ms", isLoggedIn, endTime - startTime);

            return isLoggedIn ? "已登录" : "未登录";
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            String errorMsg = "海信SSO登录状态检查失败: " + e.getMessage();
            log.error("海信SSO登录状态检查失败，耗时: {} ms", endTime - startTime, e);
            return errorMsg;
        }
    }

    /**
     * 检查当前会话是否已登录
     * 
     * @return true表示已登录且会话有效，false表示未登录或会话已过期
     */
    private boolean isSessionLoggedIn() {
        // 检查是否存在共享上下文
        if (getUSerContext() == null) {
            return false;
        }

        // 检查登录是否过期
        long currentTime = System.currentTimeMillis();
        if (currentTime - lastLoginTime > LOGIN_VALIDITY_PERIOD) {
            log.debug("会话已过期，上次登录时间: {}，当前时间: {}", lastLoginTime, currentTime);
            return false;
        }

        return true;
    }

    /**
     * 验证当前会话是否仍然有效
     * 通过访问一个需要登录的页面来验证会话状态
     * 
     * @param testUrl 用于验证会话的测试页面URL
     * @return true表示会话有效，false表示会话无效
     */
    private boolean validateSession(String testUrl) {
        if (!isSessionLoggedIn()) {
            return false;
        }

        try {
            Page page = getUSerContext().newPage();
            try {
                page.navigate(testUrl, new Page.NavigateOptions().setWaitUntil(WaitUntilState.NETWORKIDLE));
                String currentUrl = page.url();

                // 如果重定向到了登录页面，说明会话已失效
                if (currentUrl.startsWith(SSO_LOGIN_URL)) {
                    log.debug("会话验证失败，已重定向到登录页面");
                    return false;
                }

                log.debug("会话验证成功，当前页面URL: {}", currentUrl);
                return true;
            } finally {
                page.close();
            }
        } catch (Exception e) {
            log.warn("会话验证过程中发生异常: {}", e.getMessage());
            return false;
        }
    }

    /**
     * 执行登录并更新登录状态
     * 
     * @param page 当前页面对象
     * @throws Exception 登录过程中的异常
     */
    private String performLoginAndUpdateStatus(Page page) throws Exception {
        log.info("开始执行SSO登录流程");

        String ssoUsername = getUserName();
        String ssoPassword = getPassword();

        try {
            // 填入用户名
            log.debug("正在定位用户名输入框: {}", USERNAME_INPUT_SELECTOR);
            Locator usernameInput = page.locator(USERNAME_INPUT_SELECTOR);
            if (usernameInput.count() == 0) {
                throw new RuntimeException("未找到用户名输入框");
            }
            usernameInput.fill(ssoUsername);
            log.debug("用户名输入完成");

            // 填入密码
            log.debug("正在定位密码输入框: {}", PASSWORD_INPUT_SELECTOR);
            Locator passwordInput = page.locator(PASSWORD_INPUT_SELECTOR);
            if (passwordInput.count() == 0) {
                throw new RuntimeException("未找到密码输入框");
            }
            passwordInput.fill(ssoPassword);
            log.debug("密码输入完成");

            // 点击登录按钮
            log.debug("正在定位登录按钮: {}", LOGIN_BUTTON_SELECTOR);
            Locator loginButton = page.locator(LOGIN_BUTTON_SELECTOR);
            if (loginButton.count() == 0) {
                throw new RuntimeException("未找到登录按钮");
            }
            loginButton.click();
            log.info("登录按钮点击完成，等待登录响应");

            // 等待页面开始跳转（表示登录请求已发送）
            page.waitForLoadState(LoadState.NETWORKIDLE);

            if (page.url().equals(SSO_MFA_URL)) {
                log.info("检测到MFA页面，自动发送验证码...");
                // 执行MFA登录，传递Context以保证生命周期管理
                String result = sendVerificationCode(ssoUsername, page, getUSerContext());
                
                log.info("已发送验证码，请查看短信");

                return result;
            }

            // 更新登录时间
            lastLoginTime = System.currentTimeMillis();
            log.info("SSO登录成功，登录时间已更新");

            return "SSO登录成功";
        } catch (Exception e) {
            log.error("SSO登录过程中发生异常", e);
            throw new RuntimeException("SSO登录失败: " + e.getMessage(), e);
        }
    }
}
