package pangea.hiagent.web.service;

import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;

import pangea.hiagent.model.ToolConfig;

import java.util.List;
import java.util.Map;

/**
 * 工具配置服务接口
 * 用于处理工具参数配置的读取和保存
 */
public interface ToolConfigService {
    
    /**
     * 根据工具名称获取参数配置（带缓存）
     * @param toolName 工具名称
     * @return 参数配置键值对
     */
    @Cacheable(value = "toolConfigByToolName", key = "#toolName")
    Map<String, String> getToolParams(String toolName);
    
    /**
     * 根据工具名称和参数名称获取参数值（带缓存）
     * @param toolName 工具名称
     * @param paramName 参数名称
     * @return 参数值
     */
    @Cacheable(value = "toolConfig", key = "#toolName + '_' + #paramName")
    String getParamValue(String toolName, String paramName);
    
    /**
     * 保存参数值（自动清除缓存）
     * @param toolName 工具名称
     * @param paramName 参数名称
     * @param paramValue 参数值
     */
    @CacheEvict(value = "toolConfig", key = "#toolName + '_' + #paramName")
    void saveParamValue(String toolName, String paramName, String paramValue);
    
    /**
     * 获取所有工具配置（带缓存）
     * @return 工具配置列表
     */
    @Cacheable(value = "allToolConfigs", key = "'all'")
    List<ToolConfig> getAllToolConfigs();
    
    /**
     * 根据工具名称和参数名称获取工具配置（带缓存）
     * @param toolName 工具名称
     * @param paramName 参数名称
     * @return 工具配置对象
     */
    @Cacheable(value = "toolConfig", key = "#toolName + '_' + #paramName")
    ToolConfig getToolConfig(String toolName, String paramName);
    
    /**
     * 保存工具配置（自动清除相关缓存）
     * @param toolConfig 工具配置对象
     * @return 保存后的工具配置对象
     */
    @CacheEvict(value = {"toolConfig", "toolConfigByToolName", "toolConfigsByToolName"}, 
                key = "#toolConfig.toolName + '_' + #toolConfig.paramName")
    ToolConfig saveToolConfig(ToolConfig toolConfig);
    
    /**
     * 删除工具配置（自动清除相关缓存）
     * @param id 配置ID
     */
    @CacheEvict(value = {"toolConfig", "toolConfigByToolName", "toolConfigsByToolName"}, 
                allEntries = true) // 删除配置时清除所有缓存，因为不知道具体工具名
    void deleteToolConfig(String id);
    
    /**
     * 根据工具名称获取工具配置列表（带缓存）
     * @param toolName 工具名称
     * @return 工具配置列表
     */
    @Cacheable(value = "toolConfigsByToolName", key = "#toolName")
    List<ToolConfig> getToolConfigsByToolName(String toolName);
}