/**
 * 日志工具函数模块
 * 提供日志记录和管理功能
 */
import { nextTick } from 'vue';

/**
 * 添加日志
 * @param message - 日志消息
 * @param type - 日志类型 (info, warn, error等)
 * @param logAreaId - 日志显示区域的ID
 */
export const addLog = (message: string, type: string, logAreaId: string = 'log-area'): void => {
  try {
    const logArea = document.getElementById(logAreaId);
    if (logArea) {
      const logItem = document.createElement('div');
      logItem.className = `log-${type}`;
      
      // 对特定错误消息进行友好化处理
      let displayMessage = message;
      if (message.includes('sso.hisense.com') && message.includes('setRequestHeader')) {
        displayMessage = `${message} (这是海信SSO系统的前端错误，不影响系统核心功能)`;
      }
      
      logItem.textContent = `[${new Date().toLocaleTimeString()}] ${displayMessage}`;
      
      // 添加数据属性以便于调试
      logItem.setAttribute('data-log-type', type);
      logItem.setAttribute('data-timestamp', new Date().toISOString());
      
      logArea.appendChild(logItem);
      
      // 限制日志数量，避免内存泄漏
      while (logArea.children.length > 100) {
        logArea.removeChild(logArea.firstChild!);
      }
      
      // 使用nextTick确保DOM更新后再滚动
      nextTick(() => {
        // 检查用户是否在查看历史日志
        const isScrolledToBottom = logArea.scrollHeight - logArea.clientHeight <= logArea.scrollTop + 10;
        if (isScrolledToBottom) {
          // 自动滚动到最新日志
          logArea.scrollTop = logArea.scrollHeight;
        }
      });
      
      // 同时输出到浏览器控制台，方便调试
      console.log(`[${type.toUpperCase()}][${new Date().toLocaleTimeString()}] ${message}`);
    } else {
      // 如果无法找到日志区域，使用console作为后备
      console.log(`[${type.toUpperCase()}][${new Date().toLocaleTimeString()}] ${message}`);
    }
  } catch (e) {
    // 即使日志记录失败，也不要影响主流程
    console.error('日志记录失败:', e);
  }
};

/**
 * 清空日志
 * @param logAreaId - 日志显示区域的ID
 */
export const clearLogs = (logAreaId: string = 'log-area'): void => {
  const logArea = document.getElementById(logAreaId);
  if (logArea) {
    logArea.innerHTML = '';
  }
};